/**
 * \file sdc_parse_arg.c
 *
 * \brief Everything related to parsing sdc_ctl arguments
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <argp.h>
#include <sdc.h>
#include <string.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include <inttypes.h>

#include "sdc_ctl.h"
#include "sdc_ctl_action.h"
#include "sdc_ctl_args.h"


error_t argp_err_exit_status = (error_t)SDC_UNKNOWN_ERROR;
const char *argp_program_version = "SDC Command Line Tool v2.0";


struct arg_operation_spec {
    __const char *name;
    struct argp *argpspec;
    action_operations_t operation;
};


static error_t sdc_clt_argp_parse_master(int key, char *arg,
                                         struct argp_state *state);

static void sdc_ctl_parse_action(const struct arg_operation_spec *op_spec, struct argp_state *state, action_t *action);
static error_t sdc_clt_argp_parse_user(const char *arg, uid_t *uid, char *error_prefix);
static error_t sdc_clt_argp_parse_group(const char *arg, gid_t *gid, char *error_prefix);
static error_t sdc_clt_argp_parse_sizet(const char *arg, size_t *l, char *error_prefix, bool gtzero);
static void sdc_ctl_initialize_state_children(const struct argp_state *state, const struct arg_operation_spec *op_spec);

static error_t sdc_clt_argp_parse_keys_common(int key, char *arg, struct argp_state *state, bool last, bool keystore, bool builtin, bool allow_mod_with_unset);
static error_t sdc_clt_argp_parse_keystore(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_keystore_and_last(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_keystore_builtin_and_last(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_keystore_builtin_and_last_allow_mod_with_unset(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_in_common(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_out_common(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_iv_common(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_tag_len_common(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_tag_common(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_dgst_len_common(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_dgst_common(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_mechanism_common(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_mechanism_extended(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_huge_data_common(int key, char *arg, struct argp_state *state);

static error_t sdc_clt_argp_parse_remove(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_install(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_non_formatted_wrap(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_non_formatted_unwrap(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_formatted_wrap(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_formatted_unwrap(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_non_formatted_encrypt(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_non_formatted_decrypt(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_formatted_encrypt(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_formatted_decrypt(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_non_formatted_sign(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_non_formatted_verify(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_formatted_sign(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_formatted_verify(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_non_formatted_dgst(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_rng(int key, char *arg, struct argp_state *state);
static error_t sdc_clt_argp_parse_arch_info(int key, char *arg, struct argp_state *state);


/* NOTE: The order of the childs will be defined locally */
enum {
    ARGP_ORDER__KEY_CHILDS=10,
    ARGP_ORDER__IN_CHILDS=20,
    ARGP_ORDER__OUT_CHILDS=30,
    ARGP_ORDER__IV_AUTO=40,
    ARGP_ORDER__IV_NOIV=41,
    ARGP_ORDER__IV_CHILDS=42,
    ARGP_ORDER__DGST_TAG_IN_CHILDS=50,
    ARGP_ORDER__DGST_TAG_LEN_IN_CHILDS=51,
    ARGP_ORDER__DGST_TAG_OUT_CHILDS=52,
    ARGP_ORDER__DGST_TAG_LEN_OUT_CHILDS=53,
    ARGP_ORDER__MECH_CHILDS=60,
    ARGP_ORDER__HUGE_DATA_CHILDS=70,

    ARGP_ORDER__INSTALL_RANDOM=20,
    ARGP_ORDER__INSTALL_PLAINFILE=21,
    ARGP_ORDER__INSTALL_WRAPFILE=22,
    ARGP_ORDER__INSTALL_BITS_BYTES=30,
    ARGP_ORDER__INSTALL_GROUP=31,
    ARGP_ORDER__INSTALL_PERM=33,
    ARGP_ORDER__INSTALL_PERSISTENT=35,
    ARGP_ORDER__INSTALL_STORE_KID=11,
    ARGP_ORDER__INSTALL_FMT=40,
};

#define KEY_PERMISSIONS_SHORT 'P'

/* kid specification */
#define KEY_LAST_KEY_SHORT 'l'
#define KEY_EXP_KID_SHORT 'k'
#define KEY_FILE_KID_SHORT 'K'
#define KEY_BUILTIN_BYTES_SHORT 'b'
#define KEY_BUILTIN_BITS_SHORT 'B'
#define KEY_BUILTIN_USER_SHORT 'u'
#define KEY_BUILTIN_GROUP_SHORT 'g'
#define KEY_BUILTIN_PRIVATE_MOD_SHORT -2
#define KEY_BUILTIN_PUBLIC_MOD_SHORT -3
#define KEY_BUILTIN_PERMISSIONS_SHORT KEY_PERMISSIONS_SHORT

#define LAST_KEYS       { "last_kid",           KEY_LAST_KEY_SHORT,             NULL,                   0,  "Use key of previous operation",                                                   1}
#define KEYSTORE_KEYS   { "kid",                KEY_EXP_KID_SHORT,              "kid",                  0,  "Specify kid explicitly",                                                          2}, \
                        { "file-kid",           KEY_FILE_KID_SHORT,             "kid-file",             0,  "Read kid from file",                                                              3}
#define BUILTIN_KEYS    { "builtin",            KEY_BUILTIN_BITS_SHORT,         "length",               0,  "Use built-in key of specified length (bits)",                                     4}, \
                        { "builtinBytes",       KEY_BUILTIN_BYTES_SHORT,        "length",               0,  "Use built-in key of specified length (bytes)",                                    4}, \
                        { "user",               KEY_BUILTIN_USER_SHORT,         "username",             0,  "User to be used when generating built-in key (default current effective user)",   5}, \
                        { "group",              KEY_BUILTIN_GROUP_SHORT,        "groupname",            0,  "Group to be used when generating built-in key (default current effective group)", 6}, \
                        { "secret-modifier",    KEY_BUILTIN_PRIVATE_MOD_SHORT,  "filename",             0,  "Use the file content as modifier when generating the builtin key. The modifier might contain secret data. " \
                                                                                                            "As a matter of fact the modifier is not stored to formatted output and automatic loading of key is not available in this case.", \
                                                                                                                                                                                               8}, \
                        { "public-modifier",    KEY_BUILTIN_PUBLIC_MOD_SHORT,   "filename",             0,  "Use the file content as modifier when generating the builtin key. The modifier must not contain any secret data. " \
                                                                                                            "In order to support autoloading of key for formatted output, the modifier is included in formatted output.", \
                                                                                                                                                                                               9}, \
                        { "permissions",        KEY_BUILTIN_PERMISSIONS_SHORT,  "permission string",    0,  "String describing the permissions of the builtin key",                            7}

static struct argp_option sdc_clt_argp_keystore[] = {
    KEYSTORE_KEYS,
    { 0 }
};
static struct argp argp_keystore = {
    sdc_clt_argp_keystore,
    sdc_clt_argp_parse_keystore,
    "",
    "",
    NULL,
    NULL,
    NULL
};
static struct argp_child argp_keystore_child[] = {
    { &argp_keystore, ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { 0 }
};

static struct argp_option sdc_clt_argp_keystore_and_last[] = {
    LAST_KEYS,
    KEYSTORE_KEYS,
    { 0 }
};
static struct argp argp_keystore_and_last = {
    sdc_clt_argp_keystore_and_last,
    sdc_clt_argp_parse_keystore_and_last,
    "",
    "",
    NULL,
    NULL,
    NULL
};
static struct argp_child argp_keystore_and_last_child[] = {
    { &argp_keystore_and_last, ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { 0 }
};

static struct argp_option sdc_clt_argp_keystore_builtin_and_last[] = {
    LAST_KEYS,
    KEYSTORE_KEYS,
    BUILTIN_KEYS,
    { 0 }
};
static struct argp argp_keystore_builtin_and_last = {
    sdc_clt_argp_keystore_builtin_and_last,
    sdc_clt_argp_parse_keystore_builtin_and_last,
    "",
    "",
    NULL,
    NULL,
    NULL
};
static struct argp argp_keystore_builtin_and_last_allow_mod_with_unset = {
    sdc_clt_argp_keystore_builtin_and_last,
    sdc_clt_argp_parse_keystore_builtin_and_last_allow_mod_with_unset,
    "",
    "",
    NULL,
    NULL,
    NULL
};


/* input output */
#define OP_COMMON_INPUT 'f'
#define OP_COMMON_OUTPUT 'o'
#define OP_COMMON_IV_FILE_SHORT 'i'
#define OP_COMMON_IV_LEN_SHORT 'I'
#define OP_COMMON_AUTO_IV_SHORT 'a'
#define OP_COMMON_NO_IV_SHORT 'n'
#define OP_COMMON_TAG_FILE_SHORT 's'
#define OP_COMMON_TAG_LEN_SHORT 'S'
#define OP_COMMON_MECHANISM_SHORT 'm'
#define OP_COMMON_MECHANISM_OPT_SHORT 'M'
#define OP_COMMON_HUGE_DATA_CHUNK_SHORT 'c'

static struct argp_option sdc_clt_argp_in_common[] = {
    { "input",       OP_COMMON_INPUT,          "filename",   0,  "Input data",                                                  1},
    { 0 }
};
static struct argp argp_in_common = {
    sdc_clt_argp_in_common,
    sdc_clt_argp_parse_in_common,
    "",
    "",
    NULL,
    NULL,
    NULL
};

static struct argp_option sdc_clt_argp_out_common[] = {
    { "output",       OP_COMMON_OUTPUT,        "filename",   0,  "Output data",                                                 1},
    { 0 }
};
static struct argp argp_out_common = {
    sdc_clt_argp_out_common,
    sdc_clt_argp_parse_out_common,
    "",
    "",
    NULL,
    NULL,
    NULL
};
static struct argp_option sdc_clt_argp_iv_common[] = {
    { "iv",          OP_COMMON_IV_FILE_SHORT,  "filename",   0,  "Read IV from file (IV length = file length)",                 1},
    { "iv-len",      OP_COMMON_IV_LEN_SHORT,   "length",     0,  "Specify the IV length",                                       2},
    { 0 }
};
static struct argp argp_iv_common = {
    sdc_clt_argp_iv_common,
    sdc_clt_argp_parse_iv_common,
    "",
    "",
    NULL,
    NULL,
    NULL
};
static struct argp_option sdc_clt_argp_tag_len_common[] = {
    { "tag-len",     OP_COMMON_TAG_LEN_SHORT,  "length",     0,  "Specify the tag length",                                1},
    { 0 }
};
static struct argp argp_tag_len_common = {
    sdc_clt_argp_tag_len_common,
    sdc_clt_argp_parse_tag_len_common,
    "",
    "",
    NULL,
    NULL,
    NULL
};
static struct argp_option sdc_clt_argp_tag_in_common[] = {
    { "tag",         OP_COMMON_TAG_FILE_SHORT, "filename",   0,  "Read tag from file (tag length = file length)",   1},
    { 0 }
};
static struct argp argp_tag_in_common = {
    sdc_clt_argp_tag_in_common,
    sdc_clt_argp_parse_tag_common,
    "",
    "",
    NULL,
    NULL,
    NULL
};
static struct argp_option sdc_clt_argp_tag_out_common[] = {
    { "tag",         OP_COMMON_TAG_FILE_SHORT, "filename",   0,  "Write tag to file",                                     1},
    { 0 }
};
static struct argp argp_tag_out_common = {
    sdc_clt_argp_tag_out_common,
    sdc_clt_argp_parse_tag_common,
    "",
    "",
    NULL,
    NULL,
    NULL
};

static struct argp_option sdc_clt_argp_mechanism_common[] = {
    { "mechanism",     OP_COMMON_MECHANISM_SHORT,     "name",   0, "Specification of the mechanism - see archinfo command for valid options",   1},
    { 0 }
};
static struct argp argp_mechanism_common = {
    sdc_clt_argp_mechanism_common,
    sdc_clt_argp_parse_mechanism_common,
    "",
    "",
    NULL,
    NULL,
    NULL
};

static struct argp_option sdc_clt_argp_mechanism_extended[] = {
    { "mechanism_opt", OP_COMMON_MECHANISM_OPT_SHORT, "BITMSK", 0, "Hexadecimal value to select an alternative operation mode for the mechanism" , 2},
    { 0 }
};
static struct argp_child argp_mechanism_extended_child[] = {
    { &argp_mechanism_common,         ARGP_IN_ORDER, "", ARGP_ORDER__MECH_CHILDS },
    { 0 }
};
static struct argp argp_mechanism_extended = {
    sdc_clt_argp_mechanism_extended,
    sdc_clt_argp_parse_mechanism_extended,
    "",
    "",
    argp_mechanism_extended_child,
    NULL,
    NULL
};


static struct argp_option sdc_clt_argp_huge_data_common[] = {
    { "chunk_len",  OP_COMMON_HUGE_DATA_CHUNK_SHORT, "bytes",   OPTION_ARG_OPTIONAL,  "Huge data processing. Data is processed using init/update/finalize API using chunks of the specified length",   1},
    { 0 }
};
static struct argp argp_huge_data_common = {
    sdc_clt_argp_huge_data_common,
    sdc_clt_argp_parse_huge_data_common,
    "",
    "",
    NULL,
    NULL,
    NULL
};

/* remove arguments
 * kid generic is included */
static struct argp_option sdc_clt_argp_remove[] = {
    { 0 }
};
static struct argp argp_remove = {
    sdc_clt_argp_remove,
    sdc_clt_argp_parse_remove,
    "",
    "Using this operation it is possible to remove keys from the key storage\v"
    "Examples :\n"
    "\tremove -l\n"
    "\tremove -k 815",
    argp_keystore_and_last_child,
    NULL,
    NULL
};

/* install arguments
 * kid generic is included */
#define INSTALL_PLAIN_FILE_SHORT 'f'
#define INSTALL_RANDOM_SHORT     'r'
#define INSTALL_WRAPPED_KEY_SHORT 'w'
#define INSTALL_PERSISTENT_SHORT 'p'
#define INSTALL_BITS_SHORT       'B'
#define INSTALL_BYTES_SHORT      'b'
#define INSTALL_GROUP_SHORT      'g'
#define INSTALL_STORE_KID        -1
#define INSTALL_FMT_SIMPLE_SYM   -2
#define INSTALL_FMT_RSA_PUB_PEM  -3
#define INSTALL_FMT_RSA_PRI_PEM  -4
#define INSTALL_FMT_RSA_PUB_DER  -5
#define INSTALL_FMT_RSA_PRI_DER  -6
#define INSTALL_PERM_SHORT       KEY_PERMISSIONS_SHORT
static struct argp_option sdc_clt_argp_install[] = {
    { "plainfromfile", INSTALL_PLAIN_FILE_SHORT,    "filename",       OPTION_NO_USAGE,  "Read the plain key secret from the file (key length = file length)", ARGP_ORDER__INSTALL_PLAINFILE},
    { "random",        INSTALL_RANDOM_SHORT,        NULL,             OPTION_NO_USAGE,  "Generate random key secret (default key length)",                    ARGP_ORDER__INSTALL_RANDOM},
    { "wrappedfromfile", INSTALL_WRAPPED_KEY_SHORT, "filename",       OPTION_NO_USAGE,  "Read the wrapped key secret from the file",                          ARGP_ORDER__INSTALL_WRAPFILE},
    { "persistent",    INSTALL_PERSISTENT_SHORT,    NULL,             0,                "Install the key persistently",                                       ARGP_ORDER__INSTALL_PERSISTENT},
    { "bits",          INSTALL_BITS_SHORT,          "bits",           0,                "Specify the key length",                                             ARGP_ORDER__INSTALL_BITS_BYTES},
    { "bytes",         INSTALL_BYTES_SHORT,         "bytes",          0,                "Specify the key length",                                             ARGP_ORDER__INSTALL_BITS_BYTES},
    { "group",         INSTALL_GROUP_SHORT,         "group",          0,                "Desired gid or group name",                                          ARGP_ORDER__INSTALL_GROUP},
    { "permissions",   INSTALL_PERM_SHORT,    "permission string",    0,                "String describing the permissions of the builtin key",               ARGP_ORDER__INSTALL_PERM},
    { "store-kid",     INSTALL_STORE_KID,           "kid-file",       0,                "Write KID to file",                                                  ARGP_ORDER__INSTALL_STORE_KID},
    { "symmetric",     INSTALL_FMT_SIMPLE_SYM,      NULL,             0,                "The key is a symmetric key (default)",                               ARGP_ORDER__INSTALL_FMT},
    { "rsa-pub-pem",   INSTALL_FMT_RSA_PUB_PEM,     NULL,             0,                "The key is a rsa public key (pem encoded)",                          ARGP_ORDER__INSTALL_FMT},
    { "rsa-pri-pem",   INSTALL_FMT_RSA_PRI_PEM,     NULL,             0,                "The key is a rsa private key (pem encoded)",                         ARGP_ORDER__INSTALL_FMT},
    { "rsa-pub-der",   INSTALL_FMT_RSA_PUB_DER,     NULL,             0,                "The key is a rsa public key (der encoded)",                          ARGP_ORDER__INSTALL_FMT},
    { "rsa-pri-der",   INSTALL_FMT_RSA_PRI_DER,     NULL,             0,                "The key is a rsa private key (der encoded)",                         ARGP_ORDER__INSTALL_FMT},
    { 0 }
};

static struct argp argp_install = {
    sdc_clt_argp_install,
    sdc_clt_argp_parse_install,
    "-r\n"
    "-f filename",
    "Using this operation it is possible to add new keys to the key storage\v"
    "In case no KID is specified a new KID is determined automatically\n"
    "Examples :\n"
    "\tinstall -k 815 -r [-b 16]\n"
    "\tinstall -f FILE",
    argp_keystore_child,
    NULL,
    NULL
};

static struct argp_option sdc_clt_argp_non_formatted_wrap[] = {
    { "auto-iv", OP_COMMON_AUTO_IV_SHORT,    "filename",       0,  "Generate IV internally and write result to file",   ARGP_ORDER__IV_AUTO},
    { "no-iv",   OP_COMMON_NO_IV_SHORT,      0,                0,  "Don't use IV",                                      ARGP_ORDER__IV_NOIV},
    { 0 }
};
static struct argp_child argp_non_formatted_wrap_child[] = {
    { &argp_keystore_builtin_and_last,  ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_out_common,                 ARGP_IN_ORDER, "", ARGP_ORDER__OUT_CHILDS },
    { &argp_tag_out_common,             ARGP_IN_ORDER, "", ARGP_ORDER__DGST_TAG_OUT_CHILDS },
    { &argp_tag_len_common,             ARGP_IN_ORDER, "", ARGP_ORDER__DGST_TAG_LEN_OUT_CHILDS },
    { &argp_iv_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IV_CHILDS },
    { &argp_mechanism_extended,         ARGP_IN_ORDER, "", ARGP_ORDER__MECH_CHILDS },
    { &argp_huge_data_common,           ARGP_IN_ORDER, "", ARGP_ORDER__HUGE_DATA_CHILDS },
    { 0 }
};
static struct argp argp_non_formatted_wrap = {
    sdc_clt_argp_non_formatted_wrap,
    sdc_clt_argp_parse_non_formatted_wrap,
    "",
    "Using this operation it is possible to wrap data using a key from the key storage\v"
    "Examples :\n"
    "\twrap -k 815 -f INPUT -o OUTPUT -s WRITE_TAG -i READ_IV\n"
    "\twrap -l -f INPUT -o OUTPUT -m AES_CCM -s WRITE_TAG --auto-iv [--iv-len=8]",
    argp_non_formatted_wrap_child,
    NULL,
    NULL
};
static struct argp_option sdc_clt_argp_non_formatted_unwrap[] = {
    { 0 }
};
static struct argp_child argp_non_formatted_unwrap_child[] = {
    { &argp_keystore_builtin_and_last,  ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_out_common,                 ARGP_IN_ORDER, "", ARGP_ORDER__OUT_CHILDS },
    { &argp_tag_in_common,              ARGP_IN_ORDER, "", ARGP_ORDER__DGST_TAG_IN_CHILDS },
    { &argp_tag_len_common,             ARGP_IN_ORDER, "", ARGP_ORDER__DGST_TAG_LEN_IN_CHILDS },
    { &argp_iv_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IV_CHILDS },
    { &argp_mechanism_extended,         ARGP_IN_ORDER, "", ARGP_ORDER__MECH_CHILDS },
    { &argp_huge_data_common,           ARGP_IN_ORDER, "", ARGP_ORDER__HUGE_DATA_CHILDS },
    { 0 }
};
static struct argp argp_non_formatted_unwrap = {
    sdc_clt_argp_non_formatted_unwrap,
    sdc_clt_argp_parse_non_formatted_unwrap,
    "",
    "Using this operation it is possible to unwrap previously wrapped data again\v"
    "Examples :\n"
    "\te.g. unwrap -k 815 -f INPUT -o OUTPUT -m AES_CCM -s READ_TAG -i READ_IV\n",
    argp_non_formatted_unwrap_child,
    NULL,
    NULL
};

static struct argp_option sdc_clt_argp_formatted_wrap[] = {
    { "auto-iv", OP_COMMON_AUTO_IV_SHORT,    NULL,       0,  "Generate IV internally",      ARGP_ORDER__IV_AUTO},
    { "no-iv",   OP_COMMON_NO_IV_SHORT,      0,          0,  "Don't use IV",                ARGP_ORDER__IV_NOIV},
    { 0 }
};
static struct argp_child argp_formatted_wrap_child[] = {
    { &argp_keystore_builtin_and_last,  ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_out_common,                 ARGP_IN_ORDER, "", ARGP_ORDER__OUT_CHILDS },
    { &argp_tag_len_common,             ARGP_IN_ORDER, "", ARGP_ORDER__DGST_TAG_LEN_OUT_CHILDS },
    { &argp_iv_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IV_CHILDS },
    { &argp_mechanism_common,           ARGP_IN_ORDER, "", ARGP_ORDER__MECH_CHILDS },
    { 0 }
};
static struct argp argp_formatted_wrap = {
    sdc_clt_argp_formatted_wrap,
    sdc_clt_argp_parse_formatted_wrap,
    "",
    "Using this operation it is possible to wrap data using a key from the key storage.\n"
    "The output is a proprietary format allowing easy unwrapping."
    "\v"
    "Examples :\n"
    "\tfwrap -k 815 -f INPUT -o OUTPUT -i READ_IV\n"
    "\tfwrap -l -f INPUT -o OUTPUT -m AES_CCM --auto-iv [--iv-len=8]",
    argp_formatted_wrap_child,
    NULL,
    NULL
};
static struct argp_option sdc_clt_argp_formatted_unwrap[] = {
    { 0 }
};
static struct argp_child argp_formatted_unwrap_child[] = {
    { &argp_keystore_builtin_and_last_allow_mod_with_unset,
      ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_out_common,                 ARGP_IN_ORDER, "", ARGP_ORDER__OUT_CHILDS },
    { 0 }
};
static struct argp argp_formatted_unwrap = {
    sdc_clt_argp_formatted_unwrap,
    sdc_clt_argp_parse_formatted_unwrap,
    "",
    "Using this operation it is possible to unwrap formatted wrapped data generated using fwrap\v"
    "In case no KID is specified the KID is read from the format (note : last KID is not updated)\n"
    "Examples :\n"
    "\te.g. funwrap -f INPUT -o OUTPUT\n",
    argp_formatted_unwrap_child,
    NULL,
    NULL
};

static struct argp_option sdc_clt_argp_non_formatted_encrypt[] = {
    { "auto-iv", OP_COMMON_AUTO_IV_SHORT,    "filename",       0,  "Generate IV internally and write result to file",   ARGP_ORDER__IV_AUTO},
    { "no-iv",   OP_COMMON_NO_IV_SHORT,      0,                0,  "Don't use IV",                                      ARGP_ORDER__IV_NOIV},
    { 0 }
};
static struct argp_child argp_non_formatted_encrypt_child[] = {
    { &argp_keystore_builtin_and_last,  ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_out_common,                 ARGP_IN_ORDER, "", ARGP_ORDER__OUT_CHILDS },
    { &argp_iv_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IV_CHILDS },
    { &argp_mechanism_extended,         ARGP_IN_ORDER, "", ARGP_ORDER__MECH_CHILDS },
    { &argp_huge_data_common,           ARGP_IN_ORDER, "", ARGP_ORDER__HUGE_DATA_CHILDS },
    { 0 }
};
static struct argp argp_non_formatted_encrypt = {
    sdc_clt_argp_non_formatted_encrypt,
    sdc_clt_argp_parse_non_formatted_encrypt,
    "",
    "Using this operation it is possible to encrypt data using a key from the key storage\v"
    "Examples :\n"
    "\tencrypt -k 815 -f INPUT -o OUTPUT -i READ_IV\n"
    "\tencrypt -l -f INPUT -o OUTPUT -m AES_CTR --auto-iv [--iv-len=8]",
    argp_non_formatted_encrypt_child,
    NULL,
    NULL
};
static struct argp_option sdc_clt_argp_non_formatted_decrypt[] = {
    { 0 }
};
static struct argp_child argp_non_formatted_decrypt_child[] = {
    { &argp_keystore_builtin_and_last,  ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_out_common,                 ARGP_IN_ORDER, "", ARGP_ORDER__OUT_CHILDS },
    { &argp_iv_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IV_CHILDS },
    { &argp_mechanism_extended,         ARGP_IN_ORDER, "", ARGP_ORDER__MECH_CHILDS },
    { &argp_huge_data_common,           ARGP_IN_ORDER, "", ARGP_ORDER__HUGE_DATA_CHILDS },
    { 0 }
};
static struct argp argp_non_formatted_decrypt = {
    sdc_clt_argp_non_formatted_decrypt,
    sdc_clt_argp_parse_non_formatted_decrypt,
    "",
    "Using this operation it is possible to decrypt previously encrypted data again\v"
    "Examples :\n"
    "\te.g. decrypt -k 815 -f INPUT -o OUTPUT -m AES_CTR -i READ_IV\n",
    argp_non_formatted_decrypt_child,
    NULL,
    NULL
};

static struct argp_option sdc_clt_argp_formatted_encrypt[] = {
    { "auto-iv", OP_COMMON_AUTO_IV_SHORT,    NULL,       0,  "Generate IV internally",  ARGP_ORDER__IV_AUTO},
    { "no-iv",   OP_COMMON_NO_IV_SHORT,      0,          0,  "Don't use IV",            ARGP_ORDER__IV_NOIV},
    { 0 }
};
static struct argp_child argp_formatted_encrypt_child[] = {
    { &argp_keystore_builtin_and_last,  ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_out_common,                 ARGP_IN_ORDER, "", ARGP_ORDER__OUT_CHILDS },
    { &argp_iv_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IV_CHILDS },
    { &argp_mechanism_common,           ARGP_IN_ORDER, "", ARGP_ORDER__MECH_CHILDS },
    { 0 }
};
static struct argp argp_formatted_encrypt = {
    sdc_clt_argp_formatted_encrypt,
    sdc_clt_argp_parse_formatted_encrypt,
    "",
    "Using this operation it is possible to encrypt data using a key from the key storage.\n"
    "The output is a proprietary format allowing easy decryption."
    "\v"
    "Examples :\n"
    "\tfencrypt -k 815 -f INPUT -o OUTPUT -i READ_IV\n"
    "\tfencrypt -l -f INPUT -o OUTPUT -m AES_CTR --auto-iv [--iv-len=8]",
    argp_formatted_encrypt_child,
    NULL,
    NULL
};
static struct argp_option sdc_clt_argp_formatted_decrypt[] = {
    { 0 }
};
static struct argp_child argp_formatted_decrypt_child[] = {
    { &argp_keystore_builtin_and_last_allow_mod_with_unset,
      ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_out_common,                 ARGP_IN_ORDER, "", ARGP_ORDER__OUT_CHILDS },
    { 0 }
};
static struct argp argp_formatted_decrypt = {
    sdc_clt_argp_formatted_decrypt,
    sdc_clt_argp_parse_formatted_decrypt,
    "",
    "Using this operation it is possible to decrypt formatted encrypted data generated using fencrypt\v"
    "In case no KID is specified the KID is read from the format (note : last KID is not updated)\n"
    "Examples :\n"
    "\te.g. fdecrypt -f INPUT -o OUTPUT\n",
    argp_formatted_decrypt_child,
    NULL,
    NULL
};

static struct argp_option sdc_clt_argp_non_formatted_sign[] = {
    { "auto-iv", OP_COMMON_AUTO_IV_SHORT,    "filename",       0,  "Generate IV internally and write result to file",   ARGP_ORDER__IV_AUTO},
    { "no-iv",   OP_COMMON_NO_IV_SHORT,      0,                0,  "Don't use IV",                                      ARGP_ORDER__IV_NOIV},
    { 0 }
};
static struct argp_child argp_non_formatted_sign_child[] = {
    { &argp_keystore_builtin_and_last,  ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_tag_out_common,             ARGP_IN_ORDER, "", ARGP_ORDER__DGST_TAG_OUT_CHILDS },
    { &argp_tag_len_common,             ARGP_IN_ORDER, "", ARGP_ORDER__DGST_TAG_LEN_OUT_CHILDS },
    { &argp_iv_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IV_CHILDS },
    { &argp_mechanism_extended,         ARGP_IN_ORDER, "", ARGP_ORDER__MECH_CHILDS },
    { &argp_huge_data_common,           ARGP_IN_ORDER, "", ARGP_ORDER__HUGE_DATA_CHILDS },
    { 0 }
};
static struct argp argp_non_formatted_sign = {
    sdc_clt_argp_non_formatted_sign,
    sdc_clt_argp_parse_non_formatted_sign,
    "",
    "Using this operation it is possible to sign data using a key from the key storage\v"
    "Examples :\n"
    "\tsign -k 815 -f INPUT -s WRITE_TAG\n"
    "\tsign -l -f INPUT -m HMAC_MD5 -s WRITE_TAG",
    argp_non_formatted_sign_child,
    NULL,
    NULL
};
static struct argp_option sdc_clt_argp_non_formatted_verify[] = {
    { 0 }
};
static struct argp_child argp_non_formatted_verify_child[] = {
    { &argp_keystore_builtin_and_last,  ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_tag_in_common,              ARGP_IN_ORDER, "", ARGP_ORDER__DGST_TAG_IN_CHILDS },
    { &argp_tag_len_common,             ARGP_IN_ORDER, "", ARGP_ORDER__DGST_TAG_LEN_IN_CHILDS },
    { &argp_iv_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IV_CHILDS },
    { &argp_mechanism_extended,         ARGP_IN_ORDER, "", ARGP_ORDER__MECH_CHILDS },
    { &argp_huge_data_common,           ARGP_IN_ORDER, "", ARGP_ORDER__HUGE_DATA_CHILDS },
    { 0 }
};
static struct argp argp_non_formatted_verify = {
    sdc_clt_argp_non_formatted_verify,
    sdc_clt_argp_parse_non_formatted_verify,
    "",
    "Using this operation it is possible to verify previously wrapped data again\v"
    "Examples :\n"
    "\te.g. verify -k 815 -f INPUT -m HMAC_MD5 -s READ_TAG\n",
    argp_non_formatted_verify_child,
    NULL,
    NULL
};

static struct argp_option sdc_clt_argp_formatted_sign[] = {
    { "auto-iv", OP_COMMON_AUTO_IV_SHORT,    NULL,       0,  "Generate IV internally",  ARGP_ORDER__IV_AUTO},
    { "no-iv",   OP_COMMON_NO_IV_SHORT,      0,          0,  "Don't use IV",            ARGP_ORDER__IV_NOIV},
    { 0 }
};
static struct argp_child argp_formatted_sign_child[] = {
    { &argp_keystore_builtin_and_last,  ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_out_common,                 ARGP_IN_ORDER, "", ARGP_ORDER__OUT_CHILDS },
    { &argp_tag_len_common,             ARGP_IN_ORDER, "", ARGP_ORDER__DGST_TAG_LEN_OUT_CHILDS },
    { &argp_iv_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IV_CHILDS },
    { &argp_mechanism_common,           ARGP_IN_ORDER, "", ARGP_ORDER__MECH_CHILDS },
    { 0 }
};
static struct argp argp_formatted_sign = {
    sdc_clt_argp_formatted_sign,
    sdc_clt_argp_parse_formatted_sign,
    "",
    "Using this operation it is possible to sign data using a key from the key storage.\n"
    "The output is a proprietary format allowing easy extraction of authenticated data (verify)."
    "\v"
    "Examples :\n"
    "\tfsign -k 815 -f INPUT -o OUTPUT\n"
    "\tfsign -l -f INPUT -o OUTPUT -m HMAC_SHA224 --auto-iv",
    argp_formatted_sign_child,
    NULL,
    NULL
};
static struct argp_option sdc_clt_argp_formatted_verify[] = {
    { 0 }
};
static struct argp_child argp_formatted_verify_child[] = {
    { &argp_keystore_builtin_and_last_allow_mod_with_unset,
      ARGP_IN_ORDER, "", ARGP_ORDER__KEY_CHILDS },
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_out_common,                 ARGP_IN_ORDER, "", ARGP_ORDER__OUT_CHILDS },
    { 0 }
};
static struct argp argp_formatted_verify = {
    sdc_clt_argp_formatted_verify,
    sdc_clt_argp_parse_formatted_verify,
    "",
    "Using this operation it is possible to extract the authenticated data from formatted data generated using fsign\v"
    "In case no KID is specified the KID is read from the format (note : last KID is not updated)\n"
    "Examples :\n"
    "\te.g. fverify -f INPUT -o OUTPUT\n",
    argp_formatted_verify_child,
    NULL,
    NULL
};


#define OP_COMMON_DGST_FILE_SHORT 'd'
#define OP_COMMON_DGST_LEN_SHORT 'D'
static struct argp_option sdc_clt_argp_non_formatted_dgst[] = {
    { 0 }
};
static struct argp_option sdc_clt_argp_dgst_len_common[] = {
    { "dgst-len",     OP_COMMON_DGST_LEN_SHORT,  "length",     0,  "Specify the digest length",                                1},
    { 0 }
};
static struct argp argp_dgst_len_common = {
    sdc_clt_argp_dgst_len_common,
    sdc_clt_argp_parse_dgst_len_common,
    "",
    "",
    NULL,
    NULL,
    NULL
};
static struct argp_option sdc_clt_argp_dgst_out_common[] = {
    { "dgst",         OP_COMMON_DGST_FILE_SHORT, "filename",   0,  "Write digest to file",                                     1},
    { 0 }
};
static struct argp argp_dgst_out_common = {
    sdc_clt_argp_dgst_out_common,
    sdc_clt_argp_parse_dgst_common,
    "",
    "",
    NULL,
    NULL,
    NULL
};
static struct argp_child argp_non_formatted_dgst_child[] = {
    { &argp_in_common,                  ARGP_IN_ORDER, "", ARGP_ORDER__IN_CHILDS },
    { &argp_dgst_out_common,            ARGP_IN_ORDER, "", ARGP_ORDER__DGST_TAG_OUT_CHILDS },
    { &argp_dgst_len_common,            ARGP_IN_ORDER, "", ARGP_ORDER__DGST_TAG_LEN_OUT_CHILDS },
    { &argp_mechanism_extended,         ARGP_IN_ORDER, "", ARGP_ORDER__MECH_CHILDS },
    { &argp_huge_data_common,           ARGP_IN_ORDER, "", ARGP_ORDER__HUGE_DATA_CHILDS },
    { 0 }
};
static struct argp argp_non_formatted_dgst = {
    sdc_clt_argp_non_formatted_dgst,
    sdc_clt_argp_parse_non_formatted_dgst,
    "",
    "Using this operation it is possible to calculate the digest of a message\v"
    "Examples :\n"
    "\tdgst -f INPUT -d WRITE_DGST\n"
    "\tdgst -f INPUT -m MD5 -d WRITE_DGST",
    argp_non_formatted_dgst_child,
    NULL,
    NULL
};


#define RNG_GEN_SHORT               'g'
#define RNG_GEN_OUTPUT_SHORT        'o'
static struct argp_option sdc_clt_argp_rng[] = {
    { "generate",           RNG_GEN_SHORT,              "bytes",          0,                "Generate ",                                                                                                       1},
    { "generate_output",    RNG_GEN_OUTPUT_SHORT,       "filename",       OPTION_NO_USAGE,  "File to store the generated output (only valid in combination with generate option)",                             2},
    { 0 }
};
static struct argp_child argp_rng_child[] = {
    { 0 }
};
static struct argp argp_rng = {
    sdc_clt_argp_rng,
    sdc_clt_argp_parse_rng,
    "",
    "Using this operation it is possible to generate random data\v"
    "Examples :\n"
    "\te.g. rng -g 40 -o output\n",
    argp_rng_child,
    NULL,
    NULL
};



#define ARCH_INFO_LIST_KEYLEN   -1
#define ARCH_INFO_LIST_WRAP     -2
#define ARCH_INFO_LIST_ENCRYPT  -3
#define ARCH_INFO_LIST_SIGNVERIFY -4
#define ARCH_INFO_LIST_DIGEST -5
#define ARCH_INFO_LIST_ERRORS -6
#define ARCH_INFO_RESOLVE_ERROR -7
static struct argp_option sdc_clt_argp_arch_info[] = {
    { "list_keylen",            ARCH_INFO_LIST_KEYLEN,      0,   OPTION_NO_USAGE,  "List all key-lengths available in this architecture",                                              1},
    { "list_wrapunwrap",        ARCH_INFO_LIST_WRAP,        0,   OPTION_NO_USAGE,  "List available combinations of algorithm and block mode for wrap/unwrap in this architecture",     5},
    { "list_encryptdecrypt",    ARCH_INFO_LIST_ENCRYPT,     0,   OPTION_NO_USAGE,  "List available combinations of algorithm and block mode for encrypt/decrypt in this architecture", 2},
    { "list_signverify",        ARCH_INFO_LIST_SIGNVERIFY,  0,   OPTION_NO_USAGE,  "List available combinations of algorithm and hash for sign/verify in this architecture",                 3},
    { "list_digest",            ARCH_INFO_LIST_DIGEST,      0,   OPTION_NO_USAGE,  "List available hash modes for digest in this architecture",                                        4},
    { "list_errors",            ARCH_INFO_LIST_ERRORS,      0,   OPTION_NO_USAGE,  "List error codes relevant in this architecture",                                                   6},
    { "get_error_string",       ARCH_INFO_RESOLVE_ERROR,    "error_code", OPTION_NO_USAGE,  "Dump error string of specific numeric error_code",                                        7},
    { 0 }
};
static struct argp argp_arch_info = {
    sdc_clt_argp_arch_info,
    sdc_clt_argp_parse_arch_info,
    "",
    "Using this action it is possible to dump architecture specific lists of supported"
    "algorithms/block modes and key lengths",
    NULL,
    NULL,
    NULL
};

static struct arg_operation_spec arg_operation_specs[] = {
    [ACTION_INSTALL] = { ACTION_NAME_INSTALL, &argp_install, ACTION_INSTALL },
    [ACTION_REMOVE] = { ACTION_NAME_REMOVE, &argp_remove, ACTION_REMOVE },
    [ACTION_NON_FORMATTED_WRAP] = { ACTION_NAME_NON_FORMATTED_WRAP, &argp_non_formatted_wrap, ACTION_NON_FORMATTED_WRAP },
    [ACTION_NON_FORMATTED_UNWRAP] = { ACTION_NAME_NON_FORMATTED_UNWRAP, &argp_non_formatted_unwrap, ACTION_NON_FORMATTED_UNWRAP },
    [ACTION_FORMATTED_WRAP] = { ACTION_NAME_FORMATTED_WRAP, &argp_formatted_wrap, ACTION_FORMATTED_WRAP },
    [ACTION_FORMATTED_UNWRAP] = { ACTION_NAME_FORMATTED_UNWRAP, &argp_formatted_unwrap, ACTION_FORMATTED_UNWRAP },
    [ACTION_NON_FORMATTED_ENCRYPT] = { ACTION_NAME_NON_FORMATTED_ENCRYPT, &argp_non_formatted_encrypt, ACTION_NON_FORMATTED_ENCRYPT },
    [ACTION_NON_FORMATTED_DECRYPT] = { ACTION_NAME_NON_FORMATTED_DECRYPT, &argp_non_formatted_decrypt, ACTION_NON_FORMATTED_DECRYPT },
    [ACTION_FORMATTED_ENCRYPT] = { ACTION_NAME_FORMATTED_ENCRYPT, &argp_formatted_encrypt, ACTION_FORMATTED_ENCRYPT },
    [ACTION_FORMATTED_DECRYPT] = { ACTION_NAME_FORMATTED_DECRYPT, &argp_formatted_decrypt, ACTION_FORMATTED_DECRYPT },
    [ACTION_NON_FORMATTED_SIGN] = { ACTION_NAME_NON_FORMATTED_SIGN, &argp_non_formatted_sign, ACTION_NON_FORMATTED_SIGN },
    [ACTION_NON_FORMATTED_VERIFY] = { ACTION_NAME_NON_FORMATTED_VERIFY, &argp_non_formatted_verify, ACTION_NON_FORMATTED_VERIFY },
    [ACTION_FORMATTED_SIGN] = { ACTION_NAME_FORMATTED_SIGN, &argp_formatted_sign, ACTION_FORMATTED_SIGN },
    [ACTION_FORMATTED_VERIFY] = { ACTION_NAME_FORMATTED_VERIFY, &argp_formatted_verify, ACTION_FORMATTED_VERIFY },
    [ACTION_NON_FORMATTED_DGST] = { ACTION_NAME_NON_FORMATTED_DGST, &argp_non_formatted_dgst, ACTION_NON_FORMATTED_DGST },
    [ACTION_RNG] = { ACTION_NAME_RNG, &argp_rng, ACTION_RNG },
    [ACTION_ARCH_INFO] = {ACTION_NAME_ARCH_INFO, &argp_arch_info, ACTION_ARCH_INFO },
};




/* top level args */
static char sdc_clt_argp_doc_master[] = "Provide adding and removing key storage keys as well as cryptographic operations using these keys\vIt is possible to concatenate multiple operation commands (each called an action). For each action additional arguments might be given.\n"
                                        "In case the user wants to apply options like --verbose to all, this option needs to be given before any operation.\n"
                                        "Examples:\n"
                                        "e.g. get options available for wrap :\n\tsdc_ctl wrap --help\n"
                                        "e.g. install a random key with random KID and remove it :\n\tsdc_ctl -v install -r -b 32 remove -l";
static char sdc_clt_argp_args_doc_master[] = "OPERATIONS\n";
#define MASTER_VERBOSE_SHORT 'v'
#define MASTER_QUIET_SHORT 'q'
static struct argp_option sdc_clt_argp_master[] = {
    { "<OPERATION> --help",              0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Show the help(e.g. arguments) of <OPERATION>", 0},
    { NULL,                              0,   NULL,       0,                              "Key management operations:",     1},
    { ACTION_NAME_INSTALL,               0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Install a key storage key",      1},
    { ACTION_NAME_REMOVE,                0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Remove a key using a key ID",    1},
    { NULL,                              0,   NULL,       0,                              "Cryptographic operations:",      10},
    { ACTION_NAME_NON_FORMATTED_WRAP,    0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Wrap data (non formatted)",      10},
    { ACTION_NAME_NON_FORMATTED_UNWRAP,  0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Unwrap data (non formatted)",    10},
    { ACTION_NAME_FORMATTED_WRAP,        0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Wrap data (formatted)",          11},
    { ACTION_NAME_FORMATTED_UNWRAP,      0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Unwrap data (formatted)",        11},
    { ACTION_NAME_NON_FORMATTED_ENCRYPT, 0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Encrypt data (non formatted)",   12},
    { ACTION_NAME_NON_FORMATTED_DECRYPT, 0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Decrypt data (non formatted)",   12},
    { ACTION_NAME_FORMATTED_ENCRYPT,     0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Encrypt data (formatted)",       13},
    { ACTION_NAME_FORMATTED_DECRYPT,     0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Decrypt data (formatted)",       13},
    { ACTION_NAME_NON_FORMATTED_SIGN,    0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Sign data (non formatted)",      14},
    { ACTION_NAME_NON_FORMATTED_VERIFY,  0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Verify data (non formatted)",    14},
    { ACTION_NAME_FORMATTED_SIGN,        0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Sign data (formatted)",          15},
    { ACTION_NAME_FORMATTED_VERIFY,      0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Verify data (formatted)",        15},
    { ACTION_NAME_NON_FORMATTED_DGST,    0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Calc digest of data",            16},
    { ACTION_NAME_RNG,                   0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Random number generation",       17},
    { NULL,                              0,   NULL,       0,                              "Information:",                   20},
    { ACTION_NAME_ARCH_INFO,             0,   NULL,       OPTION_DOC | OPTION_NO_USAGE,   "Architecture specific info",     20},
    { NULL,                              0,   NULL,       0,                              "Generic options:",               30},
    { "verbose",      MASTER_VERBOSE_SHORT,   NULL,       0,                              "Produce verbose output",         30},
    { "quiet",          MASTER_QUIET_SHORT,   NULL,       0,                              "Produce no output",              30},
    { 0 }
};
static struct argp argp_master = {
    sdc_clt_argp_master,
    sdc_clt_argp_parse_master,
    sdc_clt_argp_args_doc_master,
    sdc_clt_argp_doc_master,
    NULL,
    NULL,
    NULL
};


/**
 *
 * Parse a single option.
 */
static error_t sdc_clt_argp_parse_master(int key, char *arg,
                                         struct argp_state *state)
{
    error_t err = 0;
    arguments_t *arguments = state->input;
    size_t idx;
    size_t specs;
    struct arg_operation_spec *op_spec;
    action_t *new_action;

    switch (key) {
    /* generic stuff */
    case MASTER_VERBOSE_SHORT:
        arguments->log_level=LOGGING_VERBOSE;
        break;
    case MASTER_QUIET_SHORT:
        arguments->log_level=LOGGING_QUIET;
        break;
    case ARGP_KEY_ARG:
        err = EINVAL;
        if (arg != NULL) {
            idx = 0;
            specs = sizeof(arg_operation_specs) / sizeof(struct arg_operation_spec);

            while (idx < specs) {
                op_spec = &arg_operation_specs[idx];

                if (strcmp(arg, op_spec->name) == 0) {

                    new_action = malloc(sizeof(action_t));
                    if (new_action == NULL) {
                        argp_failure(state, 1, ENOMEM, 0);
                    } else {
                        memset(new_action, 0, sizeof(action_t));
                        new_action->operation = op_spec->operation;
                        new_action->name = op_spec->name;
                        new_action->next = NULL;
                        if (arguments->first_action == NULL) {
                            /* this is the first action */
                            arguments->first_action = new_action;
                        } else {
                            /* append to last_action */
                            arguments->last_action->next = new_action;
                        }
                        arguments->last_action = new_action;

                        sdc_ctl_parse_action(op_spec, state, new_action);
                        err = 0;
                        break;
                    }
                }
                idx++;
            }
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static void sdc_ctl_parse_action(const struct arg_operation_spec *op_spec, struct argp_state *state, action_t *action)
{
    error_t err;
    int argc = (state->argc - state->next) + 1;
    char** argv = &state->argv[state->next - 1];
    char*  argv0 =  argv[0];

    argv[0] = malloc(strlen(state->name) + strlen(op_spec->name) + 2);

    if(!argv[0])
        argp_failure(state, 1, ENOMEM, 0);

    sprintf(argv[0], "%s %s", state->name, op_spec->name);

    err = argp_parse(op_spec->argpspec, argc, argv, ARGP_IN_ORDER, &argc, action);
    if (err != 0)
        argp_failure(state, 1, err, 0);

    free(argv[0]);

    argv[0] = argv0;

    state->next += argc - 1;

    return;
}

static error_t sdc_clt_argp_parse_user(const char *arg, uid_t *uid, char *error_prefix)
{
    struct passwd *user;
    char *endptr = NULL;

    if (!arg) {
        return EINVAL;
    }

    /* try interpreting the argument as string name first */
    user = getpwnam(arg);
    if (user != NULL) {
        /* group has been found */
        *uid = user->pw_uid;;
    } else {
        /* try interpreting the argument as a number */
        *uid = strtoul (arg, &endptr, 10);
        if ((*arg==0) || (*endptr != 0)) {
            fprintf(stderr, "%s: Failed to resolve user\n", error_prefix);
            return EINVAL;
        }
    }

    return 0;
}

static error_t sdc_clt_argp_parse_group(const char *arg, gid_t *gid, char *error_prefix)
{
    struct group *group;
    char *endptr = NULL;

    if (!arg) {
        return EINVAL;
    }

    /* try interpreting the argument as string name first */
    group = getgrnam(arg);
    if (group != NULL) {
        /* group has been found */
        *gid = group->gr_gid;
    } else {
        /* try interpreting the argument as a number */
        *gid = strtoul (arg, &endptr, 10);
        if ((*arg==0) || (*endptr != 0)) {
            fprintf(stderr, "%s: Failed to resolve group\n", error_prefix);
            return EINVAL;
        }
    }

    return 0;
}

static error_t sdc_clt_argp_parse_sizet(const char *arg, size_t *l, char *error_prefix, bool gtzero)
{
    char *endptr = NULL;

    if (!arg) {
        return EINVAL;
    }

    *l = strtoul (arg, &endptr, 10);
    if ((*arg==0) || (*endptr != 0) || ((*l == 0) && gtzero)) {
        fprintf(stderr, "%s: Failed to parse sizet\n", error_prefix);
        return EINVAL;
    }

    return 0;
}

static void sdc_ctl_initialize_state_children(const struct argp_state *state, const struct arg_operation_spec *op_spec)
{
    size_t children;
    size_t idx;

    __const struct argp_child *child;

    child = op_spec->argpspec->children;
    children = 0;
    while (child->argp) {
        children++;
        child++;
    }

    for (idx = 0; idx < children; idx++) {
        state->child_inputs[idx] = state->input;
    }
}


static error_t sdc_clt_argp_parse_keys_common(int key, char *arg, struct argp_state *state, bool last, bool keystore, bool builtin, bool allow_mod_with_unset)
{
    error_t err = 0;
    char *endptr = NULL;
    action_t *action = state->input;
    FILE *fp;
    size_t len;
    sdc_error_t sdcerr;

    switch (key) {
    case ARGP_KEY_INIT:
        action->key_spec.source = ACTION_KEY_UNSPECIFIED;
        action->key_spec.keystore.kid = SDC_FLAG_INVALID_KID;
        action->key_spec.builtin.uid_set = false;
        action->key_spec.builtin.gid_set = false;
        /*action->key_spec.builtin.len_set = false;*/
        action->key_spec.builtin.permissions = NULL;

        action->key_spec.builtin.secret_modifier = true;
        action->key_spec.builtin.modifier_filename = NULL;
        break;
    case KEY_LAST_KEY_SHORT:
        if (last) {
            if (action->key_spec.source != ACTION_KEY_UNSPECIFIED) {
                fprintf(stderr, "Key is already specified\n");
                err = EINVAL;
            } else {
                action->key_spec.source = ACTION_KEY_LAST;
            }
        } else {
            err = ARGP_ERR_UNKNOWN;
        }
        break;
    case KEY_EXP_KID_SHORT:
        if (keystore) {
            if (action->key_spec.source != ACTION_KEY_UNSPECIFIED) {
                fprintf(stderr, "Key is already specified\n");
                err = EINVAL;
            } else {
                action->key_spec.source = ACTION_KID_ARG;
                if (!arg) {
                    err = EINVAL;
                } else {
                    char kid_string[11] = {0}; /* Max number of digits for uint32_t is not more than 11 */
                    uint32_t arg_kid;
                    int ret;
                    arg_kid = strtoul (arg, &endptr, 10);
                    snprintf(kid_string, sizeof(kid_string), "%u", arg_kid);
                    ret = strcmp(arg, kid_string);
                    if(ret == 0) {
                        action->key_spec.keystore.kid = arg_kid;
                    } else {
                        action->key_spec.keystore.kid = SDC_FLAG_INVALID_KID;
                    }

                    if ((*arg==0) || (*endptr != 0) || (action->key_spec.keystore.kid == SDC_FLAG_INVALID_KID)) {
                        fprintf(stderr, "%s: Failed to parse kid\n", state->name);
                        err = EINVAL;
                    }
                }
            }
        } else {
            err = ARGP_ERR_UNKNOWN;
        }
        break;
    case KEY_FILE_KID_SHORT:
        if (keystore) {
            if (action->key_spec.source != ACTION_KEY_UNSPECIFIED) {
                fprintf(stderr, "Key is already specified\n");
                err = EINVAL;
            } else {
                action->key_spec.source = ACTION_KID_ARG;
                err = EINVAL;
                if (arg != NULL) {
                    fp = fopen (arg, "r");
                    if (fp != NULL) {
                        if (1 == fscanf(fp, "%u", &action->key_spec.keystore.kid))
                            err = 0;

                        fclose (fp);
                    }
                }
            }
        } else {
            err = ARGP_ERR_UNKNOWN;
        }
        break;
    case KEY_BUILTIN_BITS_SHORT:
    case KEY_BUILTIN_BYTES_SHORT:
        if (builtin) {
            if (action->key_spec.source != ACTION_KEY_UNSPECIFIED) {
                fprintf(stderr, "Key is already specified\n");
                err = EINVAL;
            } else {
                err = sdc_clt_argp_parse_sizet(arg, &len, state->name, true);
                if (err == 0) {
                    if (key == KEY_BUILTIN_BITS_SHORT)
                        sdcerr = sdc_key_len_from_bits(len, &action->key_spec.builtin.len);
                    else
                        sdcerr = sdc_key_len_from_bytes(len, &action->key_spec.builtin.len);
                    if (sdcerr != SDC_OK) {
                        fprintf(stderr, "Invalid length for builtin key\n");
                        err = EINVAL;
                    }
                }
                action->key_spec.source = ACTION_BUILTIN_ARG;
            }
        } else {
            err = ARGP_ERR_UNKNOWN;
        }
        break;
    case KEY_BUILTIN_USER_SHORT:
        if (builtin) {
            if (action->key_spec.builtin.uid_set) {
                fprintf(stderr, "Built-in key user is already specified\n");
                err = EINVAL;
            } else {
                err = sdc_clt_argp_parse_user(arg, &action->key_spec.builtin.uid, state->name);
                action->key_spec.builtin.uid_set = true;
            }
        } else {
            err = ARGP_ERR_UNKNOWN;
        }
        break;
    case KEY_BUILTIN_GROUP_SHORT:
        if (builtin) {
            if (action->key_spec.builtin.gid_set) {
                fprintf(stderr, "Built-in key group is already specified\n");
                err = EINVAL;
            } else {
                err = sdc_clt_argp_parse_group(arg, &action->key_spec.builtin.gid, state->name);
                action->key_spec.builtin.gid_set = true;
            }
        } else {
            err = ARGP_ERR_UNKNOWN;
        }
        break;
    case KEY_BUILTIN_PUBLIC_MOD_SHORT:
        action->key_spec.builtin.secret_modifier = false;
    /* FALLTHROUGH */
    case KEY_BUILTIN_PRIVATE_MOD_SHORT:
        if (builtin) {
            if (action->key_spec.builtin.modifier_filename) {
                fprintf(stderr, "Built-in modifier is already specified\n");
                err = EINVAL;
            } else {
                action->key_spec.builtin.modifier_filename = arg;
            }
        } else {
            err = ARGP_ERR_UNKNOWN;
        }
        break;
    case KEY_BUILTIN_PERMISSIONS_SHORT:
        if (builtin) {
            if (action->key_spec.builtin.permissions) {
                fprintf(stderr, "Built-in permissions are already specified\n");
                err = EINVAL;
            } else {
                action->key_spec.builtin.permissions = arg;
            }
        } else {
            err = ARGP_ERR_UNKNOWN;
        }
        break;
    case ARGP_KEY_SUCCESS:
        if (action->key_spec.source != ACTION_BUILTIN_ARG) {
            if ((action->key_spec.builtin.uid_set) || (action->key_spec.builtin.gid_set)) {
                fprintf(stderr, "You can't specify uid or gid without selecting built-in key\n");
                err = EINVAL;
            }
            /* for formatted unprotect (unwrap, verify, decrypt) we might need to specify
             * a secret modifier even if key is unset(autoload the key from formatted)
             */
            if ((!allow_mod_with_unset) ||
                (action->key_spec.source != ACTION_KEY_UNSPECIFIED) ||
                (!action->key_spec.builtin.secret_modifier))
                if (action->key_spec.builtin.modifier_filename) {
                    fprintf(stderr, "You can't specify a modifier without selecting built-in key\n");
                    err = EINVAL;
                }
            if (action->key_spec.builtin.permissions) {
                fprintf(stderr, "You can't specify permissions without selecting built-in key\n");
                err = EINVAL;
            }
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_keystore(int key, char *arg,
                                           struct argp_state *state)
{
    return sdc_clt_argp_parse_keys_common(key, arg, state, false, true, false, false);
}

static error_t sdc_clt_argp_parse_keystore_and_last(int key, char *arg,
                                                    struct argp_state *state)
{
    return sdc_clt_argp_parse_keys_common(key, arg, state, true, true, false, false);
}

static error_t sdc_clt_argp_parse_keystore_builtin_and_last(int key, char *arg, struct argp_state *state)
{
    return sdc_clt_argp_parse_keys_common(key, arg, state, true, true, true, false);
}

static error_t sdc_clt_argp_parse_keystore_builtin_and_last_allow_mod_with_unset(int key, char *arg, struct argp_state *state)
{
    return sdc_clt_argp_parse_keys_common(key, arg, state, true, true, true, true);
}

static error_t sdc_clt_argp_parse_in_common(int key, char *arg, struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    switch (key) {
    case ARGP_KEY_INIT:
        action->input = NULL;
        break;
    case OP_COMMON_INPUT:
        if (action->input) {
            fprintf(stderr, "%s: Input already specified\n", state->name);
            err = EINVAL;
        }
        action->input = arg;
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_out_common(int key, char *arg, struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    switch (key) {
    case ARGP_KEY_INIT:
        action->output = NULL;
        break;
    case OP_COMMON_OUTPUT:
        if (action->output) {
            fprintf(stderr, "%s: Output already specified\n", state->name);
            err = EINVAL;
        }
        action->output = arg;
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_iv_common(int key, char *arg, struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    switch (key) {
    case ARGP_KEY_INIT:
        action->iv_spec.iv_mode = ACTION_IV_DEFAULT;
        action->iv_spec.iv_filename = NULL;
        action->iv_spec.iv_len = 0;
        action->iv_spec.iv_len_specified = false;
        break;
    case OP_COMMON_IV_FILE_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_FILE;
        action->iv_spec.iv_filename = arg;
        break;
    case OP_COMMON_IV_LEN_SHORT:
        err = sdc_clt_argp_parse_sizet(arg, &action->iv_spec.iv_len, state->name, false);
        action->iv_spec.iv_len_specified = true;
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_tag_len_common(int key, char *arg, struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    switch (key) {
    case ARGP_KEY_INIT:
        action->dgst_tag_spec.dgst_tag_len = 0;
        action->dgst_tag_spec.dgst_tag_len_specified = false;
        break;
    case OP_COMMON_TAG_LEN_SHORT:
        err = sdc_clt_argp_parse_sizet(arg, &action->dgst_tag_spec.dgst_tag_len, state->name, false);
        action->dgst_tag_spec.dgst_tag_len_specified = true;
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_tag_common(int key, char *arg, struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    switch (key) {
    case ARGP_KEY_INIT:
        action->dgst_tag_spec.dgst_tag_filename = NULL;
        break;
    case OP_COMMON_TAG_FILE_SHORT:
        if (action->dgst_tag_spec.dgst_tag_filename) {
            fprintf(stderr, "%s: Tag already specified\n", state->name);
            err = EINVAL;
        }
        action->dgst_tag_spec.dgst_tag_filename = arg;
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_dgst_len_common(int key, char *arg, struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    switch (key) {
    case ARGP_KEY_INIT:
        action->dgst_tag_spec.dgst_tag_len = 0;
        action->dgst_tag_spec.dgst_tag_len_specified = false;
        break;
    case OP_COMMON_DGST_LEN_SHORT:
        err = sdc_clt_argp_parse_sizet(arg, &action->dgst_tag_spec.dgst_tag_len, state->name, false);
        action->dgst_tag_spec.dgst_tag_len_specified = true;
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_dgst_common(int key, char *arg, struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    switch (key) {
    case ARGP_KEY_INIT:
        action->dgst_tag_spec.dgst_tag_filename = NULL;
        break;
    case OP_COMMON_DGST_FILE_SHORT:
        if (action->dgst_tag_spec.dgst_tag_filename) {
            fprintf(stderr, "%s: Digest already specified\n", state->name);
            err = EINVAL;
        }
        action->dgst_tag_spec.dgst_tag_filename = arg;
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_mechanism_common(int key, char *arg, struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    switch (key) {
    case ARGP_KEY_INIT:
        action->mech_spec.mechanism = NULL;
        break;
    case OP_COMMON_MECHANISM_SHORT:
        if (action->mech_spec.mechanism) {
            fprintf(stderr, "%s: Mechanism already specified\n", state->name);
            err = EINVAL;
        }
        action->mech_spec.mechanism = arg;
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_mechanism_extended(int key, char *arg, struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;
    int parsed_params;
    char end;
    size_t children;
    size_t idx;

    __const struct argp_child *child;

    switch (key) {
    case ARGP_KEY_INIT:
        /* initalize children */
        child = argp_mechanism_extended.children;
        children = 0;
        while (child->argp) {
            children++;
            child++;
        }

        for (idx = 0; idx < children; idx++) {
            state->child_inputs[idx] = state->input;
        }

        /* initialize own content */
        action->mech_spec.mechanism_opt = 0;
        action->mech_spec.mechanism_opt_set = false;
        break;
    case OP_COMMON_MECHANISM_OPT_SHORT:
        if (action->mech_spec.mechanism_opt_set) {
            fprintf(stderr, "%s: Mechanism option already specified\n", state->name);
            err = EINVAL;
        }

        /* try parsing hexstring
         * trailing character is just there to detect an unexpected string
         * expected number of elements is always 1
         */
        if (arg) {
            if (*arg=='=')
                arg++;

            parsed_params = sscanf(arg, "0x%" SCNx64 "%c",&action->mech_spec.mechanism_opt, &end);
            if (parsed_params != 1) {
                fprintf(stderr, "%s: Failed to parse mechanism option hexstring (e.g. 0x0021)\n", state->name);
                err = EINVAL;
            } else {
                action->mech_spec.mechanism_opt_set = true;
            }
        } else {
            fprintf(stderr, "%s: Missing parameter for mechanism option\n", state->name);
            err = EINVAL;
        }
        break;
    case ARGP_KEY_SUCCESS:
        if ((action->mech_spec.mechanism_opt_set) && (action->mech_spec.mechanism == NULL))
        {
            fprintf(stderr, "%s: Mechanism option can't be used without selection the mechanism explicitly\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_huge_data_common(int key, char *arg, struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;
    char *endptr = NULL;

    switch (key) {
    case ARGP_KEY_INIT:
        action->huge_data_opts.enable = false;
        action->huge_data_opts.chunk_length_set = false;
        break;
    case OP_COMMON_HUGE_DATA_CHUNK_SHORT:
        if (action->huge_data_opts.enable) {
            fprintf(stderr, "%s: Huge data mode is already enabled\n", state->name);
            err = EINVAL;
        }

        action->huge_data_opts.enable = true;

        if (arg) {
            action->huge_data_opts.chunk_length_set = true;

            /* skip initial = */
            if (*arg=='=')
                arg++;

            action->huge_data_opts.chunk_length = strtoul (arg, &endptr, 10);
            if ((*arg==0) || (*endptr != 0)) {
                fprintf(stderr, "%s: Failed to parse chunk length\n", state->name);
                return EINVAL;
            }

            if (action->huge_data_opts.chunk_length == 0) {
                fprintf(stderr, "%s: Invalid chunk length\n", state->name);
                return EINVAL;
            }
        }

        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_remove(int key, char *arg,
                                         struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_REMOVE]);
        break;
    case ARGP_KEY_SUCCESS:
        if (action->key_spec.source == ACTION_KEY_UNSPECIFIED) {
            fprintf(stderr, "%s : No KID specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_install(int key, char *arg,
                                          struct argp_state *state)
{
    sdc_error_t sdcerr;
    error_t err = 0;
    size_t len;
    action_t *action = state->input;

    switch (key) {
    case ARGP_KEY_INIT:
        action->options.install_options.stor_opt = SDC_VOLATILE_STORAGE_KEY;
        action->options.install_options.type = ACTION_INSTALL_KEY_UNKNOWN;
        action->options.install_options.gid = (gid_t)-1;
        action->options.install_options.fmt = SDC_KEY_FMT_UNKNOWN;
        action->options.install_options.len = SDC_KEY_LEN_UNKNOWN;
        action->options.install_options.enc = SDC_KEY_ENC_UNKNOWN;
        action->options.install_options.kid_file = NULL;
        action->options.install_options.permissions = NULL;

        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_INSTALL]);

        break;
    case INSTALL_PERSISTENT_SHORT:
        action->options.install_options.stor_opt = SDC_PERSISTENT_STORAGE_KEY;
        break;
    case INSTALL_GROUP_SHORT:
        err = sdc_clt_argp_parse_group(arg, &action->options.install_options.gid, state->name);
        break;
    case INSTALL_STORE_KID:
        action->options.install_options.kid_file = arg;
        break;
    case INSTALL_PERM_SHORT:
        if (action->options.install_options.permissions) {
            fprintf(stderr, "Install permissions are already specified\n");
            err = EINVAL;
        } else {
            action->options.install_options.permissions = arg;
        }
        break;
    case INSTALL_PLAIN_FILE_SHORT:
        if (action->options.install_options.type != ACTION_INSTALL_KEY_UNKNOWN) {
            fprintf(stderr, "%s: -%c, -%c or -%c must not be given more than once\n", state->name, INSTALL_RANDOM_SHORT, INSTALL_PLAIN_FILE_SHORT, INSTALL_WRAPPED_KEY_SHORT);
            err = EINVAL;
        }
        action->options.install_options.type = ACTION_INSTALL_PLAIN_KEY;
        action->input = arg;
        if (!arg) {
            err = EINVAL;
        }
        break;
    case INSTALL_WRAPPED_KEY_SHORT:
        if (action->options.install_options.type != ACTION_INSTALL_KEY_UNKNOWN) {
            fprintf(stderr, "%s: -%c, -%c or -%c must not be given more than once\n", state->name, INSTALL_RANDOM_SHORT, INSTALL_PLAIN_FILE_SHORT, INSTALL_WRAPPED_KEY_SHORT);
            err = EINVAL;
        }
        action->options.install_options.type = ACTION_INSTALL_WRAPPED_KEY;
        action->input = arg;
        if (!arg) {
            err = EINVAL;
        }
        break;
    case INSTALL_RANDOM_SHORT:
        if (action->options.install_options.type != ACTION_INSTALL_KEY_UNKNOWN) {
            fprintf(stderr, "%s: -%c, -%c or -%c must not be given more than once\n", state->name, INSTALL_RANDOM_SHORT, INSTALL_PLAIN_FILE_SHORT, INSTALL_WRAPPED_KEY_SHORT);
            err = EINVAL;
        }
        action->options.install_options.type = ACTION_INSTALL_RANDOM_KEY;
        break;
    case INSTALL_BITS_SHORT:
    case INSTALL_BYTES_SHORT:
        if (action->options.install_options.len != SDC_KEY_LEN_UNKNOWN) {
            err = ARGP_ERR_UNKNOWN;
        } else {
            err = sdc_clt_argp_parse_sizet(arg, &len, state->name, true);
            if (err == 0) {
                if (key == INSTALL_BITS_SHORT)
                    sdcerr = sdc_key_len_from_bits(len, &action->options.install_options.len);
                else
                    sdcerr = sdc_key_len_from_bytes(len, &action->options.install_options.len);

                if (sdcerr != SDC_OK) {
                    fprintf(stderr, "Invalid length for install\n");
                    err = EINVAL;
                }
            }
        }
        break;
    case INSTALL_FMT_SIMPLE_SYM:
    case INSTALL_FMT_RSA_PUB_PEM:
    case INSTALL_FMT_RSA_PRI_PEM:
    case INSTALL_FMT_RSA_PUB_DER:
    case INSTALL_FMT_RSA_PRI_DER:
        if (action->options.install_options.fmt != SDC_KEY_FMT_UNKNOWN) {
            err = ARGP_ERR_UNKNOWN;
        } else {
            switch(key) {
            case INSTALL_FMT_RSA_PUB_PEM:
                action->options.install_options.fmt = SDC_KEY_FMT_RSA_PUBLIC;
                action->options.install_options.enc = SDC_KEY_ENC_PEM;
                break;
            case INSTALL_FMT_RSA_PRI_PEM:
                action->options.install_options.fmt = SDC_KEY_FMT_RSA_PRIVATE;
                action->options.install_options.enc = SDC_KEY_ENC_PEM;
                break;
            case INSTALL_FMT_RSA_PUB_DER:
                action->options.install_options.fmt = SDC_KEY_FMT_RSA_PUBLIC;
                action->options.install_options.enc = SDC_KEY_ENC_DER;
                break;
            case INSTALL_FMT_RSA_PRI_DER:
                action->options.install_options.fmt = SDC_KEY_FMT_RSA_PRIVATE;
                action->options.install_options.enc = SDC_KEY_ENC_DER;
                break;
            default:
                action->options.install_options.fmt = SDC_KEY_FMT_SIMPLE_SYM;
                action->options.install_options.enc = SDC_KEY_ENC_PLAIN;
                break;
            }
        }
        break;
    case ARGP_KEY_SUCCESS:
        if (action->options.install_options.type == ACTION_INSTALL_KEY_UNKNOWN) {
            fprintf(stderr, "%s: You need to specify if the key is read (-%c), generated (-%c) or wrapped (-%c)\n", state->name, INSTALL_PLAIN_FILE_SHORT, INSTALL_RANDOM_SHORT, INSTALL_WRAPPED_KEY_SHORT);
            err = EINVAL;
        }
        if ((action->options.install_options.len == SDC_KEY_LEN_UNKNOWN) &&
            ((action->options.install_options.fmt == SDC_KEY_FMT_RSA_PRIVATE) ||
             (action->options.install_options.fmt == SDC_KEY_FMT_RSA_PUBLIC))) {
            fprintf(stderr, "%s:Public private keys require explicitly specified key length\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_non_formatted_wrap(int key, char *arg,
                                                     struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_NON_FORMATTED_WRAP]);
        break;
    case OP_COMMON_AUTO_IV_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_AUTO_GEN_WRITE_FILE;
        action->iv_spec.iv_filename = arg;
        break;
    case OP_COMMON_NO_IV_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_NOIV;
        break;
    case ARGP_KEY_SUCCESS:
        if (action->key_spec.source == ACTION_KEY_UNSPECIFIED) {
            fprintf(stderr, "%s : No KID specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->output) {
            fprintf(stderr, "%s : No output specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->dgst_tag_spec.dgst_tag_filename) {
            fprintf(stderr, "%s : No tag specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_non_formatted_unwrap(int key, char *arg,
                                                       struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_NON_FORMATTED_UNWRAP]);
        break;
    case ARGP_KEY_SUCCESS:
        if (action->key_spec.source == ACTION_KEY_UNSPECIFIED) {
            fprintf(stderr, "%s : No KID specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->output) {
            fprintf(stderr, "%s : No output specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->dgst_tag_spec.dgst_tag_filename) {
            fprintf(stderr, "%s : No tag specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_formatted_wrap(int key, char *arg,
                                                 struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_FORMATTED_WRAP]);
        break;
    case OP_COMMON_AUTO_IV_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_AUTO_GEN_NO_FILE;
        break;
    case OP_COMMON_NO_IV_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_NOIV;
        break;
    case ARGP_KEY_SUCCESS:
        if (action->key_spec.source == ACTION_KEY_UNSPECIFIED) {
            fprintf(stderr, "%s : No KID specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->output) {
            fprintf(stderr, "%s : No output specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_formatted_unwrap(int key, char *arg,
                                                   struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_FORMATTED_UNWRAP]);
        break;
    case ARGP_KEY_SUCCESS:
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->output) {
            fprintf(stderr, "%s : No output specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_non_formatted_encrypt(int key, char *arg,
                                                        struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_NON_FORMATTED_ENCRYPT]);
        break;
    case OP_COMMON_AUTO_IV_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_AUTO_GEN_WRITE_FILE;
        action->iv_spec.iv_filename = arg;
        break;
    case OP_COMMON_NO_IV_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_NOIV;
        break;
    case ARGP_KEY_SUCCESS:
        if (action->key_spec.source == ACTION_KEY_UNSPECIFIED) {
            fprintf(stderr, "%s : No KID specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->output) {
            fprintf(stderr, "%s : No output specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_non_formatted_decrypt(int key, char *arg,
                                                        struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_NON_FORMATTED_DECRYPT]);
        break;
    case ARGP_KEY_SUCCESS:
        if (action->key_spec.source == ACTION_KEY_UNSPECIFIED) {
            fprintf(stderr, "%s : No KID specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->output) {
            fprintf(stderr, "%s : No output specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_formatted_encrypt(int key, char *arg,
                                                    struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_FORMATTED_ENCRYPT]);
        break;
    case OP_COMMON_AUTO_IV_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_AUTO_GEN_NO_FILE;
        break;
    case OP_COMMON_NO_IV_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_NOIV;
        break;
    case ARGP_KEY_SUCCESS:
        if (action->key_spec.source == ACTION_KEY_UNSPECIFIED) {
            fprintf(stderr, "%s : No KID specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->output) {
            fprintf(stderr, "%s : No output specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_formatted_decrypt(int key, char *arg,
                                                    struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_FORMATTED_DECRYPT]);
        break;
    case ARGP_KEY_SUCCESS:
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->output) {
            fprintf(stderr, "%s : No output specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}


static error_t sdc_clt_argp_parse_non_formatted_sign(int key, char *arg,
                                                     struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_NON_FORMATTED_SIGN]);
        break;
    case OP_COMMON_AUTO_IV_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_AUTO_GEN_WRITE_FILE;
        action->iv_spec.iv_filename = arg;
        break;
    case OP_COMMON_NO_IV_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_NOIV;
        break;
    case ARGP_KEY_SUCCESS:
        if (action->key_spec.source == ACTION_KEY_UNSPECIFIED) {
            fprintf(stderr, "%s : No KID specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->dgst_tag_spec.dgst_tag_filename) {
            fprintf(stderr, "%s : No tag specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_non_formatted_verify(int key, char *arg,
                                                       struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_NON_FORMATTED_VERIFY]);
        break;
    case ARGP_KEY_SUCCESS:
        if (action->key_spec.source == ACTION_KEY_UNSPECIFIED) {
            fprintf(stderr, "%s : No KID specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->dgst_tag_spec.dgst_tag_filename) {
            fprintf(stderr, "%s : No tag specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_formatted_sign(int key, char *arg,
                                                 struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_FORMATTED_SIGN]);
        break;
    case OP_COMMON_AUTO_IV_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_AUTO_GEN_NO_FILE;
        break;
    case OP_COMMON_NO_IV_SHORT:
        if (action->iv_spec.iv_mode != ACTION_IV_DEFAULT) {
            fprintf(stderr, "%s: IV mode already specified\n", state->name);
            err = EINVAL;
        }
        action->iv_spec.iv_mode = ACTION_IV_NOIV;
        break;
    case ARGP_KEY_SUCCESS:
        if (action->key_spec.source == ACTION_KEY_UNSPECIFIED) {
            fprintf(stderr, "%s : No KID specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->output) {
            fprintf(stderr, "%s : No output specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_formatted_verify(int key, char *arg,
                                                   struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_FORMATTED_VERIFY]);
        break;
    case ARGP_KEY_SUCCESS:
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->output) {
            fprintf(stderr, "%s : No output specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_non_formatted_dgst(int key, char *arg,
                                                     struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_NON_FORMATTED_DGST]);
        break;
    case ARGP_KEY_SUCCESS:
        if (!action->input) {
            fprintf(stderr, "%s : No input specified!\n", state->name);
            err = EINVAL;
        }
        if (!action->dgst_tag_spec.dgst_tag_filename) {
            fprintf(stderr, "%s : No digest specified!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

static error_t sdc_clt_argp_parse_rng(int key, char *arg,
                                      struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;

    (void)arg;

    switch (key) {
    case ARGP_KEY_INIT:
        action->input = NULL;
        action->output = NULL;
        action->options.rng_options.gen = false;
        action->options.rng_options.gen_len = 0;

        sdc_ctl_initialize_state_children (state, &arg_operation_specs[ACTION_RNG]);
        break;
    case RNG_GEN_OUTPUT_SHORT:
        if (action->output) {
            fprintf(stderr, "%s: Generator output already specified!\n", state->name);
            err = EINVAL;
        }
        action->output = arg;
        break;
    case RNG_GEN_SHORT:
        if (action->options.rng_options.gen) {
            fprintf(stderr, "%s: Generate option already specified once!\n", state->name);
            err = EINVAL;
        } else {
            err = sdc_clt_argp_parse_sizet(arg, &action->options.rng_options.gen_len, state->name, true);
        }
        action->options.rng_options.gen = true;
        break;
    case ARGP_KEY_SUCCESS:
        if (action->options.rng_options.gen) {
            if (action->options.rng_options.gen_len <= 0) {
                fprintf(stderr, "%s : Invalid length for generation!\n", state->name);
                err = EINVAL;
            }
            if (!action->output) {
                fprintf(stderr, "%s : No output specified!\n", state->name);
                err = EINVAL;
            }
        } else {
            if (action->output) {
                fprintf(stderr, "%s : Output is only valid for generate!\n", state->name);
                err = EINVAL;
            }
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}


static error_t sdc_clt_argp_parse_arch_info(int key, char *arg,
                                            struct argp_state *state)
{
    error_t err = 0;
    action_t *action = state->input;
    action_option_arch_info_t *opt = &action->options.arch_info_options;
    char *endptr = NULL;

    switch (key) {
    case ARGP_KEY_INIT:
        opt->dump_arch_key_lengths = false;
        opt->dump_wrap_alg_blk = false;
        opt->dump_encrypt_alg_blk = false;
        opt->dump_signverify_alg_hash = false;
        opt->dump_digest_hash = false;
        opt->dump_error_list = false;
        opt->dump_error_code = false;
        break;
    case ARCH_INFO_LIST_KEYLEN:
        opt->dump_arch_key_lengths = true;
        break;
    case ARCH_INFO_LIST_WRAP:
        opt->dump_wrap_alg_blk = true;
        break;
    case ARCH_INFO_LIST_ENCRYPT:
        opt->dump_encrypt_alg_blk = true;
        break;
    case ARCH_INFO_LIST_SIGNVERIFY:
        opt->dump_signverify_alg_hash = true;
        break;
    case ARCH_INFO_LIST_DIGEST:
        opt->dump_digest_hash = true;
        break;
    case ARCH_INFO_LIST_ERRORS:
        opt->dump_error_list = true;
        break;
    case ARCH_INFO_RESOLVE_ERROR:
        opt->dump_error_code = true;
        opt->error_code = (sdc_error_t)strtoul (arg, &endptr, 10);
        if ((*arg==0) || (*endptr != 0)) {
            fprintf(stderr, "Failed to parse error code\n");
            return EINVAL;
        }
        break;
    case ARGP_KEY_SUCCESS:
        if (
            !opt->dump_arch_key_lengths &&
            !opt->dump_wrap_alg_blk &&
            !opt->dump_encrypt_alg_blk &&
            !opt->dump_signverify_alg_hash &&
            !opt->dump_digest_hash &&
            !opt->dump_error_list &&
            !opt->dump_error_code
            ) {
            fprintf(stderr, "%s : No option selected!\n", state->name);
            err = EINVAL;
        }
        break;
    default:
        err = ARGP_ERR_UNKNOWN;
    }

    return err;
}

sdc_error_t sdc_ctl_parse_args (int argc, char **argv, arguments_t *arguments)
{
    sdc_error_t err = SDC_OK;

    /* Default values. */
    arguments->log_level = LOGGING_NORMAL;
    arguments->first_action = NULL;
    arguments->last_action = NULL;


    if (0 != argp_parse(&argp_master, argc, argv, ARGP_IN_ORDER, 0, arguments)) {
        fprintf(stderr, "Error parsing options\n");
        err = SDC_UNKNOWN_ERROR;
    }

    return err;
}
